(function () {
  const state = {
    settings: {
      autoFill: true,
      autoCapture: true,
      suggestionsLimit: 5
    },
    overlay: null,
    toast: null,
    suggestions: [],
    anchor: null,
    autoFillPerformed: false
  };

  const STYLE = `
    .gpw-overlay {
      position: absolute;
      z-index: 2147483647;
      background: #101828;
      color: #f8fafc;
      border-radius: 8px;
      box-shadow: 0 8px 20px rgba(15, 23, 42, 0.35);
      padding: 8px;
      font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
      min-width: 240px;
    }
    .gpw-overlay.hidden {
      display: none !important;
    }
    .gpw-overlay h4 {
      margin: 0 0 6px 0;
      font-size: 13px;
      font-weight: 600;
    }
    .gpw-suggestion {
      display: flex;
      flex-direction: column;
      gap: 2px;
      border: 1px solid rgba(148, 163, 184, 0.4);
      border-radius: 6px;
      padding: 6px 8px;
      margin-bottom: 6px;
      cursor: pointer;
      transition: border-color 0.2s, background 0.2s;
    }
    .gpw-suggestion:last-child {
      margin-bottom: 0;
    }
    .gpw-suggestion:hover {
      border-color: #38bdf8;
      background: rgba(56, 189, 248, 0.15);
    }
    .gpw-suggestion__site {
      font-size: 12px;
      color: #94a3b8;
    }
    .gpw-suggestion__user {
      font-size: 13px;
      font-weight: 600;
      color: #f8fafc;
    }
    .gpw-list {
      max-height: 240px;
      overflow-y: auto;
      scrollbar-width: thin;
      scrollbar-color: #475569 #1e293b;
    }
    .gpw-list::-webkit-scrollbar {
      width: 6px;
    }
    .gpw-list::-webkit-scrollbar-track {
      background: #1e293b;
      border-radius: 3px;
    }
    .gpw-list::-webkit-scrollbar-thumb {
      background: #475569;
      border-radius: 3px;
    }
    .gpw-list::-webkit-scrollbar-thumb:hover {
      background: #64748b;
    }
    .gpw-toast {
      position: fixed;
      left: 50%;
      bottom: 24px;
      transform: translateX(-50%);
      background: #0f172a;
      color: #f8fafc;
      padding: 10px 16px;
      border-radius: 999px;
      box-shadow: 0 10px 35px rgba(15, 23, 42, 0.35);
      font-size: 13px;
      z-index: 2147483647;
      opacity: 0;
      transition: opacity 0.25s ease-in-out;
    }
    .gpw-toast.visible {
      opacity: 1;
    }
    .gpw-flash-fill {
      animation: gpw-field-flash 1.9s ease-out;
      border: 3px solid rgba(34, 197, 94, 0.95) !important;
      background-color: transparent !important;
      box-shadow: 0 0 0 0 rgba(34, 197, 94, 0.45) !important;
    }
    @keyframes gpw-field-flash {
      0% {
        box-shadow: 0 0 0 0 rgba(34, 197, 94, 0.45);
      }
      60% {
        box-shadow: 0 0 0 14px rgba(34, 197, 94, 0);
      }
      100% {
        box-shadow: 0 0 0 22px rgba(34, 197, 94, 0);
      }
    }
  `;

  const USERNAME_SELECTORS = [
    // Atributo autocomplete estándar
    'input[autocomplete="username"]',
    'input[autocomplete="email"]',
    // Nombres comunes para usuario
    'input[name*="user" i]',
    'input[name*="login" i]',
    'input[name*="email" i]',
    'input[name*="dni" i]',
    'input[name*="nif" i]',
    'input[name*="ident" i]',
    'input[name*="account" i]',
    'input[name*="member" i]',
    'input[name*="phone" i]',
    'input[name*="telefono" i]',
    'input[name*="mobile" i]',
    'input[name*="cedula" i]',
    'input[name*="documento" i]',
    // IDs comunes
    'input[id*="user" i]',
    'input[id*="login" i]',
    'input[id*="email" i]',
    'input[id*="dni" i]',
    'input[id*="nif" i]',
    'input[id*="ident" i]',
    'input[id*="account" i]',
    'input[id*="phone" i]',
    // Placeholders comunes
    'input[placeholder*="usuario" i]',
    'input[placeholder*="user" i]',
    'input[placeholder*="email" i]',
    'input[placeholder*="correo" i]',
    'input[placeholder*="dni" i]',
    'input[placeholder*="teléfono" i]',
    'input[placeholder*="telefono" i]',
    'input[placeholder*="móvil" i]',
    'input[placeholder*="movil" i]',
    // Tipos específicos
    'input[type="email"]',
    'input[type="tel"]',
    // Fallback: cualquier input de texto visible (usar con precaución en búsqueda global)
    'input[type="text"]'
  ];

  function isUsernameField(field) {
    if (!(field instanceof HTMLInputElement)) return false;
    if (field.type === 'password' || field.type === 'submit' || field.type === 'hidden') return false;
    if (field.disabled || field.readOnly) return false;

    // Comprobar todos los selectores excepto el genérico input[type="text"]
    // para evitar falsos positivos en cualquier campo de texto
    const specificSelectors = USERNAME_SELECTORS.filter(s => s !== 'input[type="text"]');
    return specificSelectors.some(selector => field.matches(selector));
  }

  async function init() {
    injectStyle();
    createOverlay();
    createToast();
    state.settings = await loadSettings();
    if (state.settings.autoFill) {
      document.addEventListener('focusin', handleFocusIn, true);
      document.addEventListener('click', handleClick, true);
      scheduleInitialAutoFill();
    }
    if (state.settings.autoCapture) {
      document.addEventListener('submit', handleFormSubmit, true);
    }
    chrome.storage.onChanged.addListener(handleStorageChange);
    document.addEventListener('keydown', handleEscape, true);
  }

  function injectStyle() {
    if (document.getElementById('gpw-style')) return;
    const style = document.createElement('style');
    style.id = 'gpw-style';
    style.textContent = STYLE;
    document.head.appendChild(style);
  }

  function createOverlay() {
    state.overlay = document.createElement('div');
    state.overlay.className = 'gpw-overlay hidden';
    state.overlay.innerHTML = '<h4>Gestor Pass</h4><div class="gpw-list"></div>';
    document.body.appendChild(state.overlay);

    document.addEventListener('mousedown', (event) => {
      if (!state.overlay || state.overlay.classList.contains('hidden')) return;
      if (!state.overlay.contains(event.target)) {
        hideOverlay();
      }
    });

    // Ocultar overlay solo cuando se hace scroll fuera del overlay (en la página)
    window.addEventListener('scroll', (event) => {
      // Si el scroll ocurre dentro del overlay, no hacer nada
      if (state.overlay && state.overlay.contains(event.target)) {
        return;
      }
      hideOverlay();
    }, true);
  }

  function createToast() {
    state.toast = document.createElement('div');
    state.toast.className = 'gpw-toast';
    document.body.appendChild(state.toast);
  }

  async function loadSettings() {
    const stored = await chrome.storage.local.get('gpwSettings');
    return {
      autoFill: true,
      autoCapture: true,
      suggestionsLimit: 5,
      ...(stored.gpwSettings || {})
    };
  }

  function handleStorageChange(changes, areaName) {
    if (areaName !== 'local' || !changes.gpwSettings) return;
    state.settings = {
      autoFill: true,
      autoCapture: true,
      suggestionsLimit: 5,
      ...(changes.gpwSettings.newValue || {})
    };
  }

  function handleFocusIn(event) {
    const target = event.target;
    if (!(target instanceof HTMLInputElement)) {
      return;
    }
    
    const isPassword = target.type.toLowerCase() === 'password';
    const isUsername = isUsernameField(target);

    if (!isPassword && !isUsername) {
      hideOverlay();
      return;
    }
    
    state.anchor = target;
    // Siempre mostrar sugerencias al hacer foco, aunque el campo tenga contenido
    requestSuggestions(false);
  }

  // También mostrar sugerencias al hacer clic (por si el campo ya tenía foco)
  function handleClick(event) {
    const target = event.target;
    if (!(target instanceof HTMLInputElement)) {
      return;
    }
    
    const isPassword = target.type.toLowerCase() === 'password';
    const isUsername = isUsernameField(target);

    if (!isPassword && !isUsername) {
      return;
    }
    
    // Si ya es el anchor actual y el overlay está oculto, volver a mostrarlo
    if (state.anchor === target && state.overlay?.classList.contains('hidden')) {
      requestSuggestions(false);
    } else if (state.anchor !== target) {
      state.anchor = target;
      requestSuggestions(false);
    }
  }

  function handleEscape(event) {
    if (event.key === 'Escape') {
      hideOverlay();
    }
  }

  function scheduleInitialAutoFill() {
    const startAttempt = () => attemptInitialAutoFill(0);
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', startAttempt, { once: true });
    } else {
      startAttempt();
    }

    // Observar cambios en el DOM para detectar nuevos campos de contraseña (multi-step logins)
    setupPasswordFieldObserver();
  }

  function setupPasswordFieldObserver() {
    // Guardar referencia a los campos de contraseña actuales
    let knownPasswordFields = new Set(getPasswordInputs());

    const observer = new MutationObserver((mutations) => {
      // Buscar nuevos campos de contraseña
      const currentPasswordFields = getPasswordInputs();
      const newFields = currentPasswordFields.filter((field) => !knownPasswordFields.has(field));

      if (newFields.length > 0) {
        // Actualizar los campos conocidos
        knownPasswordFields = new Set(currentPasswordFields);
        
        // Intentar autocompletar los nuevos campos (especialmente útil para multi-step)
        handleNewPasswordFields(newFields);
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true,
      attributes: true,
      attributeFilter: ['type'] // Detectar cambios de tipo (ej: text -> password)
    });
  }

  async function handleNewPasswordFields(newFields) {
    const domain = window.location.hostname;
    if (!domain) return;

    // Comprobar si hay un usuario pendiente de un paso anterior
    const pendingUsername = await getPendingUsername(domain);
    if (!pendingUsername) return;

    try {
      const response = await sendMessage('gpw:getSuggestions', { domain, force: false });
      const items = Array.isArray(response?.items) ? response.items : [];
      if (!items.length) return;

      // Buscar la credencial que coincida con el usuario del paso anterior
      const normalizedPending = pendingUsername.trim().toLowerCase();
      const matchingItem = items.find((i) => {
        const itemUsername = (i.username || '').trim().toLowerCase();
        return itemUsername === normalizedPending;
      });

      if (matchingItem) {
        // Rellenar los nuevos campos de contraseña con la credencial correcta
        newFields.forEach((field) => {
          if (!field.value) {
            field.value = matchingItem.password || '';
            field.dispatchEvent(new Event('input', { bubbles: true }));
            field.dispatchEvent(new Event('change', { bubbles: true }));
            flashField(field);
          }
        });

        // Incrementar uso
        if (matchingItem.id) {
          sendMessage('gpw:incrementUsage', { payload: { id: matchingItem.id, domain } }).catch(() => {});
        }
      }
    } catch (error) {
      console.warn('[Gestor Pass] Error al autocompletar nuevo campo de contraseña', error);
    }
  }

  async function attemptInitialAutoFill(retry) {
    if (state.autoFillPerformed) {
      return;
    }

    const passwordFields = getPasswordInputs();
    const hasEmptyField = passwordFields.some((field) => !field.value);

    if (!passwordFields.length || !hasEmptyField) {
      if (retry < 5) {
        setTimeout(() => attemptInitialAutoFill(retry + 1), 250 * (retry + 1));
      }
      return;
    }

    const domain = window.location.hostname;
    if (!domain) {
      return;
    }

    try {
      const response = await sendMessage('gpw:getSuggestions', { domain, force: false });
      const items = Array.isArray(response?.items) ? response.items : [];
      if (!items.length) {
        return;
      }

      // Comprobar si hay un usuario pendiente de un paso anterior (multi-step login)
      const pendingUsername = await getPendingUsername(domain);
      let item = items[0]; // Por defecto, el primero (ya ordenado por uso)

      if (pendingUsername) {
        // Buscar la credencial que coincida con el usuario del paso anterior
        const normalizedPending = pendingUsername.trim().toLowerCase();
        const matchingItem = items.find((i) => {
          const itemUsername = (i.username || '').trim().toLowerCase();
          return itemUsername === normalizedPending;
        });
        if (matchingItem) {
          item = matchingItem;
        }
      }

      let filledAny = false;
      passwordFields.forEach((field) => {
        filledAny = fillCredentials(item, field, { skipIfPopulated: true, highlight: true }) || filledAny;
      });

      if (filledAny) {
        state.autoFillPerformed = true;
      } else if (retry < 5) {
        setTimeout(() => attemptInitialAutoFill(retry + 1), 250 * (retry + 1));
      }
    } catch (error) {
      console.warn('[Gestor Pass] No se pudo autocompletar automáticamente', error);
    }
  }

  function getPasswordInputs() {
    return Array.from(document.querySelectorAll('input[type="password"]')).filter((input) => {
      if (!(input instanceof HTMLInputElement)) {
        return false;
      }
      if (input.disabled || input.readOnly) {
        return false;
      }
      return true;
    });
  }

  function flashField(field) {
    if (!(field instanceof HTMLElement)) {
      return;
    }
    field.classList.remove('gpw-flash-fill');
    // Force reflow so the animation restarts if needed
    void field.offsetWidth;
    field.classList.add('gpw-flash-fill');
    setTimeout(() => field.classList.remove('gpw-flash-fill'), 2200);
  }

  async function handleFormSubmit(event) {
    if (!state.settings.autoCapture) {
      return;
    }
    const form = event.target;
    if (!(form instanceof HTMLFormElement)) {
      return;
    }

    const creds = extractFormCredentials(form);
    const domain = window.location.hostname;

    // Caso 1: Formulario con usuario pero SIN contraseña (primer paso de login multi-step)
    if (creds.username && !creds.password) {
      await stashPendingUsername(domain, creds.username);
      return;
    }

    // Si no hay contraseña, no hay nada que guardar
    if (!creds.password) {
      return;
    }

    // Caso 2: Formulario con contraseña pero SIN usuario visible (segundo paso de login multi-step)
    let username = creds.username;
    if (!username) {
      const pending = await getPendingUsername(domain);
      if (pending) {
        username = pending;
        // Limpiar el usuario pendiente tras usarlo
        await clearPendingUsername(domain);
      }
    }

    // Buscar si ya existe una credencial para este usuario en este dominio
    try {
      const existingItem = await findExistingCredential(domain, username);

      if (existingItem) {
        // Si usuario y contraseña coinciden exactamente, no hacer nada
        if (existingItem.password === creds.password) {
          return;
        }

        // Si el usuario existe pero la contraseña es diferente, preguntar si actualizar
        const confirmUpdate = window.confirm(
          `La contraseña para "${username || '(sin usuario)'}" ha cambiado. ¿Deseas actualizarla en Gestor Pass?`
        );
        if (!confirmUpdate) {
          return;
        }

        try {
          await sendMessage('gpw:updateCredential', {
            payload: {
              id: existingItem.id,
              password: creds.password,
              domain: domain
            }
          });
          showToast('Contraseña actualizada en Gestor Pass');
        } catch (error) {
          showToast(error.message || 'No se pudo actualizar la contraseña', true);
        }
        return;
      }
    } catch {
      // Si falla la comprobación, continuamos con el flujo de guardado normal
    }

    // Caso 3: Credencial nueva, preguntar si guardar
    const confirmSave = window.confirm('¿Guardar esta credencial en Gestor Pass?');
    if (!confirmSave) {
      return;
    }

    try {
      await sendMessage('gpw:saveCredential', {
        payload: {
          site: window.location.origin,
          username: username,
          password: creds.password,
          notes: creds.notes,
          domain: domain
        }
      });
      showToast('Credencial guardada en Gestor Pass');
    } catch (error) {
      showToast(error.message || 'No se pudo guardar la credencial', true);
    }
  }

  // Almacena temporalmente el usuario para logins de múltiples pasos
  async function stashPendingUsername(domain, username) {
    const key = 'gpwPendingUser';
    const stored = await chrome.storage.local.get(key);
    const data = stored[key] || {};
    data[domain] = {
      username: username,
      timestamp: Date.now()
    };
    await chrome.storage.local.set({ [key]: data });
  }

  // Recupera el usuario pendiente para un dominio (si existe y no ha expirado)
  async function getPendingUsername(domain) {
    const key = 'gpwPendingUser';
    const stored = await chrome.storage.local.get(key);
    const data = stored[key] || {};
    const entry = data[domain];

    if (!entry) {
      return null;
    }

    // Expirar después de 5 minutos
    const MAX_AGE_MS = 5 * 60 * 1000;
    if (Date.now() - entry.timestamp > MAX_AGE_MS) {
      await clearPendingUsername(domain);
      return null;
    }

    return entry.username;
  }

  // Limpia el usuario pendiente para un dominio
  async function clearPendingUsername(domain) {
    const key = 'gpwPendingUser';
    const stored = await chrome.storage.local.get(key);
    const data = stored[key] || {};
    delete data[domain];
    await chrome.storage.local.set({ [key]: data });
  }

  function extractFormCredentials(form) {
    const passwordField = form.querySelector('input[type="password"]');
    const usernameField = findUsernameField(form);
    const password = passwordField?.value || '';
    const username = usernameField?.value?.trim() || '';
    const notes = form.action && form.action !== window.location.href ? `Form action: ${form.action}` : '';
    return { username, password, notes };
  }

  function findUsernameField(scope) {
    for (const selector of USERNAME_SELECTORS) {
      const field = scope.querySelector(selector);
      if (field && field instanceof HTMLInputElement && field.type !== 'password' && !field.disabled && !field.readOnly) {
        return field;
      }
    }
    return null;
  }

  /**
   * Busca una credencial existente para el dominio y usuario dados.
   * Devuelve el item completo si existe, o null si no hay coincidencia.
   */
  async function findExistingCredential(domain, username) {
    if (!domain) {
      return null;
    }

    const response = await sendMessage('gpw:getSuggestions', { domain, force: false });
    const items = Array.isArray(response?.items) ? response.items : [];

    const normalizedUsername = (username || '').trim().toLowerCase();

    // Buscar una entrada con el mismo usuario (case-insensitive)
    for (const item of items) {
      const itemUsername = (item.username || '').trim().toLowerCase();
      if (itemUsername === normalizedUsername) {
        return item;
      }
    }

    return null;
  }

  async function requestSuggestions(force) {
    const domain = window.location.hostname;
    if (!domain) return;
    try {
      const response = await sendMessage('gpw:getSuggestions', { domain, force });
      state.suggestions = response?.items || [];
      renderOverlay();
    } catch (error) {
      if (force) {
        showToast(error.message || 'No se pudieron cargar sugerencias', true);
      }
      hideOverlay();
    }
  }

  function renderOverlay() {
    if (!state.overlay || !state.anchor) return;
    // Mostrar todas las sugerencias disponibles (el scroll se encarga si hay muchas)
    const items = state.suggestions;
    if (!items.length) {
      hideOverlay();
      return;
    }

    const list = document.createElement('div');
    list.className = 'gpw-list';
    items.forEach((item) => {
      const element = document.createElement('div');
      element.className = 'gpw-suggestion';
      element.innerHTML = `
        <span class="gpw-suggestion__site">${item.site || item.domain || 'Sitio sin nombre'}</span>
        <span class="gpw-suggestion__user">${item.username || '(sin usuario)'}</span>
      `;
      element.addEventListener('mousedown', (event) => {
        event.preventDefault();
        fillCredentials(item);
      });
      list.appendChild(element);
    });

    state.overlay.innerHTML = '<h4>Gestor Pass</h4>';
    state.overlay.appendChild(list);

    const rect = state.anchor.getBoundingClientRect();
    state.overlay.style.top = `${window.scrollY + rect.bottom + 6}px`;
    state.overlay.style.left = `${window.scrollX + rect.left}px`;
    state.overlay.classList.remove('hidden');
  }

  function hideOverlay() {
    if (state.overlay) {
      state.overlay.classList.add('hidden');
    }
    state.anchor = null;
  }

  function fillCredentials(item, targetField = state.anchor, { skipIfPopulated = false, highlight = false, trackUsage = true } = {}) {
    if (!targetField) {
      return false;
    }

    const formScope = targetField.form || document;
    
    // Identificar si el campo objetivo es de contraseña o de usuario
    const isTargetPassword = targetField.type === 'password';
    
    let passwordField = null;
    let usernameField = null;

    if (isTargetPassword) {
      passwordField = targetField;
      usernameField = findUsernameField(formScope);
    } else {
      // Si el objetivo no es password, asumimos que es el de usuario (o similar)
      usernameField = targetField;
      // Intentar buscar un campo de password asociado
      passwordField = formScope.querySelector('input[type="password"]');
    }

    let usernameChanged = false;
    let passwordChanged = false;

    if (usernameField && (!skipIfPopulated || !usernameField.value)) {
      usernameField.value = item.username || '';
      usernameField.dispatchEvent(new Event('input', { bubbles: true }));
      usernameField.dispatchEvent(new Event('change', { bubbles: true }));
      usernameChanged = true;
    }

    // Solo rellenar la contraseña si existe el campo
    if (passwordField && (!skipIfPopulated || !passwordField.value)) {
      passwordField.value = item.password || '';
      passwordField.dispatchEvent(new Event('input', { bubbles: true }));
      passwordField.dispatchEvent(new Event('change', { bubbles: true }));
      passwordChanged = true;
    }

    if (highlight) {
      if (usernameChanged && usernameField) {
        flashField(usernameField);
      }
      if (passwordChanged && passwordField) {
        flashField(passwordField);
      }
    }

    if (targetField === state.anchor) {
      hideOverlay();
    }

    // Incrementar contador de uso si se rellenó algo y tenemos el id del item
    if (trackUsage && (usernameChanged || passwordChanged) && item.id) {
      const domain = window.location.hostname;
      sendMessage('gpw:incrementUsage', { payload: { id: item.id, domain } }).catch(() => {
        // Ignorar errores de tracking, no es crítico
      });
    }

    return usernameChanged || passwordChanged;
  }

  function showToast(message, isError = false) {
    if (!state.toast) return;
    state.toast.textContent = message;
    state.toast.style.background = isError ? '#b91c1c' : '#0f172a';
    state.toast.classList.add('visible');
    setTimeout(() => {
      state.toast?.classList.remove('visible');
    }, 2500);
  }

  function sendMessage(type, payload = {}) {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ type, ...payload }, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        if (!response) {
          reject(new Error('No se recibió respuesta del servicio'));
          return;
        }
        if (response.ok) {
          resolve(response.data);
        } else {
          reject(new Error(response.error || 'Acción rechazada'));
        }
      });
    });
  }

  init().catch((error) => {
    console.error('[Gestor Pass] No se pudo inicializar el content script', error);
  });
})();

