document.addEventListener('DOMContentLoaded', () => {
  const form = document.getElementById('settingsForm');
  const apiBase = document.getElementById('apiBase');
  const email = document.getElementById('email');
  const password = document.getElementById('password');
  const autoFill = document.getElementById('autoFill');
  const autoCapture = document.getElementById('autoCapture');
  const suggestionsLimit = document.getElementById('suggestionsLimit');
  const loginButton = document.getElementById('loginButton');
  const testButton = document.getElementById('testConnection');
  const message = document.getElementById('message');

  init().catch((error) => showMessage(error.message || 'No se pudo cargar la configuración', true));

  async function init() {
    const settings = await sendMessage('gpw:getSettings');
    apiBase.value = settings.apiBaseUrl || 'https://pass.gzutil.com';
    email.value = settings.email || '';
    password.value = '';
    autoFill.checked = settings.autoFill !== false;
    autoCapture.checked = settings.autoCapture !== false;
    suggestionsLimit.value = settings.suggestionsLimit || 5;

    form.addEventListener('submit', handleSubmit);
    loginButton.addEventListener('click', handleLogin);
    testButton.addEventListener('click', handleTest);
  }

  async function handleSubmit(event) {
    event.preventDefault();
    const payload = {
      apiBaseUrl: apiBase.value.trim().replace(/\/+$/, ''),
      email: email.value.trim(),
      autoFill: autoFill.checked,
      autoCapture: autoCapture.checked,
      suggestionsLimit: clamp(parseInt(suggestionsLimit.value, 10) || 5, 1, 20)
    };

    if (!payload.apiBaseUrl) {
      showMessage('Completa la URL base.', true);
      return;
    }

    try {
      await sendMessage('gpw:setSettings', { payload });
      showMessage('Configuración guardada correctamente.');
    } catch (error) {
      showMessage(error.message || 'No se pudieron guardar los cambios', true);
    }
  }

  async function handleLogin() {
    const currentEmail = email.value.trim();
    const currentPassword = password.value;

    if (!currentEmail) {
      showMessage('Introduce tu usuario (email) para iniciar sesión.', true);
      return;
    }
    if (!currentPassword || currentPassword.length < 8) {
      showMessage('Introduce una contraseña válida (mínimo 8 caracteres).', true);
      return;
    }

    loginButton.disabled = true;
    showMessage('Iniciando sesión…');
    try {
      await sendMessage('gpw:loginWithCredentials', {
        email: currentEmail,
        password: currentPassword
      });
      password.value = '';
      showMessage('Inicio de sesión correcto. Token configurado.');
    } catch (error) {
      showMessage(error.message || 'No se pudo iniciar sesión', true);
    } finally {
      loginButton.disabled = false;
    }
  }

  async function handleTest() {
    testButton.disabled = true;
    showMessage('Probando conexión…');
    try {
      await sendMessage('gpw:testConnection');
      showMessage('Conexión OK. La API respondió correctamente.');
    } catch (error) {
      showMessage(error.message || 'No se pudo contactar con la API', true);
    } finally {
      testButton.disabled = false;
    }
  }

  function clamp(value, min, max) {
    return Math.max(min, Math.min(max, value));
  }

  function showMessage(text, isError = false) {
    message.textContent = text;
    message.style.color = isError ? '#f87171' : '#a5f3fc';
  }

  function sendMessage(type, payload = {}) {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({ type, ...payload }, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        if (!response) {
          reject(new Error('Sin respuesta del servicio'));
          return;
        }
        if (response.ok) {
          resolve(response.data);
        } else {
          reject(new Error(response.error || 'Acción rechazada'));
        }
      });
    });
  }
});

